//---------------------------------------------------------------------------------------------
// E2925A system tests
// upgraded to support E2926A system test (64 bit tests)
// original code:            Thomas Dippon
// reworked for E2926A:      Tilmann Wendel
//---------------------------------------------------------------------------------------------
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <memory.h>
#include <windows.h>

#include <mini_api.h>
#include <ppr.h>
#include <conio.h>
#include <time.h>                    //CBB 8/4/97
#include "b_ntmem.h"

// define an error macro to make programming life easier
#define C(x) if ((err = (x)) != B_E_OK) \
	{ printf("%s (line %d)\n", BestErrorStringGet(err), __LINE__); }

// necessary defines
#define HEWLETT_PACKARD_VEN_ID 0x103c
#define E2925A_DEV_ID 0x2925
#define E2926A_DEV_ID 0x2926

#define STATUS_MASTER_ACTIVE            0x01
#define STATUS_PROTOCOL_ERROR           0x10
#define STATUS_DATA_COMPARE_ERROR       0x20

#define MY_ATTR_PAGE    2
#define MAX_CARDS       15
#define MAX_THREADS     15

#define USE             1
#define DONT_USE        0

#define USE_SAC         0            // use single address cycles, do NOT use Dual Address Cycles
#define USE_DAC         1            // use Dual Address Cycles


// the following struct contains all information that is needed to work with a card
// it exists once per card
typedef struct card_info_s {
  int busy;               // indicates whether the device is busy
  int usable;             // indicates if we use this card at all
  int useDAC;             // indicates whether the memory space is located above 4 Gig (B_DAC_NO,B_DAC_YES)
  struct thread_info_s *tip; // points to the thread that owns this card

  b_int32 devid;          // device ID returned by BestDevIdentifierGet
  b_handletype handle;    // handle from BestOpen
  b_hwtype usedboard;     // the hardware behind the handle
  
  b_int32 memaddr;        // memory base address of on-board memory (physical)
  b_int32 memaddr_hi;     // high address, just in case it is a 64 bit address
  b_int32 memsize;        // size of memory
  void *memptr;           // pointer to access on-board memory (virtual addr.)
                          // so far NT supports only 32 bit memory space, a 32 bit void pointer is
                          // ok (up until NT is supporting 64 bit memory space...)
  
  b_int32 ioaddr;         // I/O base address
  b_int32 iosize;         // size of I/O space
  void *ioptr;            // pointer to access on-board I/O space (virtual addr.)
  
  void *sysmem_ptr;       // pointer to allocated system memory
  b_int32 sysmem_addr;    // physical address of allocated system memory

} card_info_t;

card_info_t card_info[MAX_CARDS];   // array with information about all cards
unsigned int n_cards = 0;           // actual number of cards located in the system

// thread state enum
typedef enum { STARTING, RUNNING, TERMINATED } thread_state_t;

typedef struct thread_info_s {
  DWORD idThread;         // ID of the thread
  HANDLE hThread;         // handle of the thread
  thread_state_t state;   // the current state of the thread
  int term_request;       // =1, request to terminate
  double maxtime;                 // runtime for this thread
  struct fct_tab_s *ftp;  // pointer to test function that this thread executes
} thread_info_t;

thread_info_t thread_info[MAX_THREADS];
unsigned int n_threads = 0;

typedef struct fct_tab_s {
  char *name;					// name of the function
  LPTHREAD_START_ROUTINE fct;	// pointer to function
} fct_tab_t;

extern fct_tab_t fct_tab[];		// declaration for test function table

//---------------------------------------------------------------------------------------------
// functions from physical.cpp
//---------------------------------------------------------------------------------------------
unsigned long virtual_to_physical(unsigned long addr);
int allocate_physical_memory(unsigned long, void **, unsigned long *);
int free_physical_memory(void *, unsigned long);
void *map_physical_address(unsigned long phys_addr_low,
						   unsigned long phys_addr_hi, // changed, so we support 64 bit addresses
						   unsigned long length,
						   unsigned int bus_number, 
						   unsigned int io);
void unmap_physical_address(void *virt_addr);
int io_access(int cmd, unsigned long addr, unsigned long *data);
int test_physical_mem();

//---------------------------------------------------------------------------------------------
// global variables
//---------------------------------------------------------------------------------------------
void *aptr = 0;                             // pointer to allocated virt. memory (0=not allocated)
unsigned long phys_addr;                    // physical address of allocated memory
const b_int32 page_size = 4096;             // size of physical memory pages
unsigned long cacheline_size = 0;            // system's cachelinesize, will be set by open_all_cards
int use_ppr = 1;                            // 1=using PPR variations 0=no variations
int write_report = 1;                       // flag for generating a PPR report
int dbg = 1;                                // dbg=0: no output, =1: verbose output
FILE *stream;                               // CBB 8/4/97
double default_maxtime = 10.0;              // default runtime for tests


// --------------------------------------------------------------------------------------------
// replaces the printf, fprintf pairs within the entire code (that is at least the plan)
// this function prints to screen and to file (and as a central output function can even
// take the output and give it to some other controlling instance)
// --------------------------------------------------------------------------------------------
void do_out(FILE* stream, char *fmt, ...)
{
  static char buffer[2048]; // we can assume, that the line will never exceed this buffer
  va_list ap;

  va_start(ap, fmt);

  vsprintf(buffer, fmt, ap);
  
  fprintf(stdout,buffer); // print it on the screen
  fprintf(stream,buffer); // print it into the log file

  va_end(ap);
  
}


//---------------------------------------------------------------------------------------------
// Set up block permutations to be used when reading or writing data.
// This routine assumes, that a connection to an E2925A or E2926A is already established.
// BestPprInit gets called from here now
//---------------------------------------------------------------------------------------------
b_errtype
setup_master_block(
				   b_handletype handle, 
				   b_int32 dir, 
				   b_int32 mem_addr, 
				   b_int32 mem_addr_hi,
				   b_int32 int_addr,
				   b_int32 size, 
				   b_int32 bpage, 
				   b_int32 compflag, 
				   b_int32 compoffs,
				   int useDAC)
{
	b_errtype err;
	b_int32 val;
	char align_str[50];
	bppr_algorithmtype alg = BPPR_ALG_PERM;
	struct tm when;
	time_t now;
	b_hwtype usedhw; // the used hardware (E2926A or E2925A)
	
	
	C(BestGetHWFromHandle(handle,&usedhw)); // gives us the hardware behind this handle
	
	time(&now);
	when = *localtime(&now);
	
	if (use_ppr)
	{
		// initialize PPR, every card gets a new PPR project
		C(BestPprInit(handle));
		//----- set up PPR block properties (writing mem_size bytes to mem_addr)
		C(BestPprBlockInit(handle));
		C(BestPprBlockPermPropDefaultSet(handle));
		C(BestPprBlockPermPropSet(handle, BPPR_BLK_DIR,         dir));
		C(BestPprBlockPermPropSet(handle, BPPR_BLK_BUSADDR,     mem_addr));
		C(BestPprBlockPermPropSet(handle, BPPR_BLK_INTADDR,     int_addr));
		C(BestPprBlockPermPropSet(handle, BPPR_BLK_NOFDWORDS,   size / 4));
		C(BestPprBlockPermPropSet(handle, BPPR_BLK_ATTRPAGE,    MY_ATTR_PAGE));
		C(BestPprBlockPermPropSet(handle, BPPR_BLK_PAGENUM,     bpage));
		C(BestPprBlockPermPropSet(handle, BPPR_BLK_PAGESIZEMAX, 32));
		C(BestPprBlockPermPropSet(handle, BPPR_BLK_FILLGAPS,    1));
		C(BestPprBlockPermPropSet(handle, BPPR_BLK_CACHELINE,   cacheline_size));
		C(BestPprBlockPermPropSet(handle, BPPR_BLK_COMPFLAG,    compflag));
		C(BestPprBlockPermPropSet(handle, BPPR_BLK_COMPOFFS,    compoffs));

		// now add some E2926A capabilities (64 bit addressing)
		if((usedhw == B_HW_E2926A || usedhw == B_HW_E2926A_DEEP) && (useDAC == USE_DAC)) 
		{
			C(BestPprBlockPermPropSet(handle, BPPR_BLK_BUSDAC, 1));// 64 bit addressing will be done
			C(BestPprBlockPermPropSet(handle, BPPR_BLK_BUSADDR_HI, mem_addr_hi));
		}
			//----- set up PPR block variations
		
		//----- should be modified according to system requirements
		C(BestPprBlockVariationDefaultSet(handle));
		C(BestPprBlockVariationSet(handle, BPPR_BLK_SIZE,  "4,8,16,32", alg));
		
		// some normal and some "odd" byte enables
		C(BestPprBlockVariationSet(handle, BPPR_BLK_BYTEN, "10,0,9,3,11", alg));
		
		// all read and commands (read, read multiple, read line, write, write & invalidate)
		C(BestPprBlockVariationSet(handle, BPPR_BLK_CMDS,  "6,12,14,7", alg));
		
		// if no cache line size is set, assume it is 8
		val = cacheline_size;
		if (val == 0)
			val = 8;
		sprintf(align_str, "(%%%ld=0),(%%%ld=4),(%%%ld=%ld)", 4*val, 4*val, 4*val, 4*val-4);
		//printf("Alignment varitions: %s\n", align_str);
		
		C(BestPprBlockVariationSet(handle, BPPR_BLK_ALIGN, align_str, alg));
		
		if (write_report)
		{
			sprintf(align_str, "ppr%02d.rpt", write_report);
			printf("Generating PPR report...(%s)\n", align_str);
			fprintf(stream, "%s Generating PPR report...(%s)\n", asctime (&when), align_str);   //CBB,8/4/97
			C(BestPprReportPropDefaultSet(handle));
			C(BestPprReportFile(handle, align_str));                                             //CBB,8/4/97
			write_report++;
		}
		
		C(BestPprBlockGenerate(handle));
		
		// free PPR resources
		C(BestPprDelete(handle));
	}
	else
	{
		if (bpage != BPPR_BLK_APPEND)
			C(BestMasterBlockPageInit(handle, bpage));
		C(BestMasterBlockPropDefaultSet(handle));
		C(BestMasterBlockPropSet(handle, B_BLK_BUSADDR, mem_addr));
		C(BestMasterBlockPropSet(handle, B_BLK_BUSCMD, dir == BPPR_DIR_WRITE ? 7 : 6));
		C(BestMasterBlockPropSet(handle, B_BLK_BYTEN, 0));
		C(BestMasterBlockPropSet(handle, B_BLK_INTADDR, int_addr));
		C(BestMasterBlockPropSet(handle, B_BLK_NOFDWORDS, size / 4));
		C(BestMasterBlockPropSet(handle, B_BLK_ATTRPAGE, MY_ATTR_PAGE));
		C(BestMasterBlockPropSet(handle, B_BLK_COMPFLAG, compflag));
		C(BestMasterBlockPropSet(handle, B_BLK_COMPOFFS, compoffs));
		if((usedhw == B_HW_E2926A || usedhw == B_HW_E2926A_DEEP) && (useDAC == USE_DAC)) 
		{
			C(BestMasterBlockPropSet(handle, B_BLK_BUSDAC, 1));
			C(BestMasterBlockPropSet(handle, B_BLK_BUSADDR_HI, mem_addr_hi));
		}
		C(BestMasterBlockProg(handle));
	}
	
	return(B_E_OK);
}

//---------------------------------------------------------------------------------------------
// Set up master attribute permutations to be used when reading or writing data.
// This routine assumes, that a connection to an E2925A is already established.
// BestPprInit gets called from here now
//---------------------------------------------------------------------------------------------

b_errtype
setup_master_attributes(b_handletype handle)
{
	b_errtype err;
	b_hwtype usedhw; // the used hardware (E2926A or E2925A)
	bppr_algorithmtype alg = BPPR_ALG_PERM;
	
	C(BestGetHWFromHandle(handle,&usedhw)); // gives us the hardware behind this handle
	
	if (use_ppr)
	{
		// initialize PPR, every card gets a new PPR project
		C(BestPprInit(handle));
		//----- set up PPR master attribute properties
		C(BestPprMAttrInit(handle));
		C(BestPprMAttrPermPropDefaultSet(handle));
		//----- in the program, only one attribute page is used
		C(BestPprMAttrPermPropSet(handle, BPPR_MA_PAGENUM, MY_ATTR_PAGE));
		C(BestPprMAttrPermPropSet(handle, BPPR_MA_PAGESIZEMAX, 8000));
		C(BestPprMAttrPermPropSet(handle, BPPR_MA_FIRSTPERM, 1));
		//----- set up PPR master attribute variations
		//----- should be modified according to system requirements
		C(BestPprMAttrVariationDefaultSet(handle));
		C(BestPprMAttrVariationSet(handle, B_M_LAST,    "1,2,4,8", alg));
		C(BestPprMAttrVariationSet(handle, B_M_WAITS,   "0,1,2,5,7", alg));
		C(BestPprMAttrVariationSet(handle, B_M_RELREQ,  "off", alg));
		C(BestPprMAttrVariationSet(handle, B_M_DPERR,   "no", alg));
		C(BestPprMAttrVariationSet(handle, B_M_DSERR,   "no", alg));
		C(BestPprMAttrVariationSet(handle, B_M_APERR,   "no", alg));
		C(BestPprMAttrVariationSet(handle, B_M_DWRPAR,  "no", alg));
		C(BestPprMAttrVariationSet(handle, B_M_AWRPAR,  "no", alg));
		C(BestPprMAttrVariationSet(handle, B_M_WAITMODE,"no", alg));
		C(BestPprMAttrVariationSet(handle, B_M_STEPMODE,"no", alg));
		
		if(usedhw == B_HW_E2926A || usedhw == B_HW_E2926A_DEEP) // now add some E2926A capabilities (64 bit,...)
		{
			C(BestPprMAttrVariationSet(handle, B_M_REQ64,   "on,on,off", alg));
			C(BestPprMAttrVariationSet(handle, B_M_DELAY,   "4,30,15,0", alg));
		}
		//----- downloading PPR attribute page
		C(BestPprMAttrGenerate(handle));
		// free PPR resources
		C(BestPprDelete(handle));
	}
	else
	{
		//----- set up a simple attribute page with zero waits
		//----- bursts as long as needed
		C(BestMasterAttrPageInit(handle, MY_ATTR_PAGE));
		C(BestMasterAttrPropDefaultSet(handle));
		C(BestMasterAttrPropSet(handle, B_M_WAITS, 0));
		C(BestMasterAttrPropSet(handle, B_M_LAST, 0));
		C(BestMasterAttrPropSet(handle, B_M_DOLOOP, 1));
		
		if(usedhw == B_HW_E2926A || usedhw == B_HW_E2926A_DEEP) // now add some E2926A capabilities (64 bit,...)
		{
			C(BestMasterAttrPropSet(handle, B_M_REQ64, 1));		// 64 bit access will be tried
			//C(BestMasterAttrPropSet(handle, B_M_TRYBACK, 1));	// Try fast back-to-back cycle
		}
		
		C(BestMasterAttrPhaseProg(handle));
	}
	return(B_E_OK);
}

//---------------------------------------------------------------------------------------------
// Set up target attribute permutations to be used when reading or writing data.
// This routine assumes, that a connection to an E2925A or E2926A is already established.
// BestPprInit gets called from here now
//---------------------------------------------------------------------------------------------
b_errtype
setup_target_attributes(b_handletype handle)
{
	b_errtype err;
	b_hwtype usedhw; // the used hardware (E2926A or E2925A)
	bppr_algorithmtype alg = BPPR_ALG_PERM;
	
	C(BestGetHWFromHandle(handle,&usedhw)); // gives us the hardware behind this handle
	
	if (use_ppr)
	{
		// initialize PPR, every card gets a new PPR project
		C(BestPprInit(handle));
		//----- set up PPR target attribute properties
		C(BestPprTAttrInit(handle));
		C(BestPprTAttrPermPropDefaultSet(handle));
		//----- in the program, only one attribute page is used
		C(BestPprTAttrPermPropSet(handle, BPPR_MA_PAGENUM, MY_ATTR_PAGE));
		C(BestPprTAttrPermPropSet(handle, BPPR_MA_PAGESIZEMAX, 8000));
		C(BestPprTAttrPermPropSet(handle, BPPR_MA_FIRSTPERM, 1));
		//----- set up PPR master attribute variations
		//----- should be modified according to system requirements
		C(BestPprTAttrVariationDefaultSet(handle));
		C(BestPprTAttrVariationSet(handle, B_T_WAITS,   "0,1,2,5,7", alg));
		C(BestPprTAttrVariationSet(handle, B_T_TERM,    "noterm,retry,disconnect", alg)); // noterm, retry, disconnect
		C(BestPprTAttrVariationSet(handle, B_T_DPERR,   "no", alg));
		C(BestPprTAttrVariationSet(handle, B_T_DSERR,   "no", alg));
		C(BestPprTAttrVariationSet(handle, B_T_APERR,   "no", alg));
		C(BestPprTAttrVariationSet(handle, B_T_WRPAR,   "no", alg));
		
		if(usedhw == B_HW_E2926A || usedhw == B_HW_E2926A_DEEP) // now add some E2926A capabilities (64 bit,...)
		{
			C(BestPprTAttrVariationSet(handle, B_T_ACK64,   "on,off,on,on", alg));
		}
		
		//----- downloading PPR attribute page
		C(BestPprTAttrGenerate(handle));
		// free PPR resources
		C(BestPprDelete(handle));
		C(BestTargetAttrPageSelect(handle, MY_ATTR_PAGE));
	}
	else
	{
		//----- set up a simple attribute page with zero waits
		//----- bursts as long as needed
		C(BestTargetAttrPageInit(handle, MY_ATTR_PAGE));
		C(BestTargetAttrPropDefaultSet(handle));
		C(BestTargetAttrPropSet(handle, B_T_WAITS, 4));
		C(BestTargetAttrPropSet(handle, B_T_TERM, B_TERM_NOTERM));
		C(BestTargetAttrPropSet(handle, B_T_DOLOOP, 1));
		
		if(usedhw == B_HW_E2926A || usedhw == B_HW_E2926A_DEEP) // now add some E2926A capabilities (64 bit,...)
		{
			C(BestTargetAttrPropSet(handle, B_T_ACK64, 1));				// for use with 2926A only
		}
		
		C(BestTargetAttrPhaseProg(handle));
		C(BestTargetAttrPageSelect(handle, MY_ATTR_PAGE));
	}
	return(B_E_OK);
}

//---------------------------------------------------------------------------------------------
// Run the master block page and wait until the transactions are completed
//---------------------------------------------------------------------------------------------

b_errtype
master_block_page_run(b_handletype handle, b_int32 bpage)
{
	b_errtype err;
	b_int32 status_reg;
	b_int32 runtime = 1000;
	
	//printf("Running card...\n");
	C(BestMasterBlockPageRun(handle, bpage));
	if(use_ppr)
	{
		C(BestPprBlockResultGet(handle,BPPR_BLK_TIME,&runtime));
	}
	
	Sleep(runtime / 1000); // wait a little before you start polling 
	
	do
	{
		C(BestStatusRegGet(handle, &status_reg));
	} while ((status_reg & STATUS_MASTER_ACTIVE) != 0);
	
	return(B_E_OK);
}



//---------------------------------------------------------------------------------------------
// check the status of a single card. If it reports a data compare or protocol error,
// print it out and return -1. Otherwise return 0
// If flag==0, print the result only if a card fails. If flag==1 print results
// unconditionally.
//---------------------------------------------------------------------------------------------
int
check_card(int flag, card_info_t *cdp)
{
	b_errtype err;
	b_handletype handle = cdp->handle;
	int ret = 0,j;
	b_int32 status,errval,errval2;
	b_int32 i = cdp - card_info;
	struct tm when;
	time_t now;
	char * perr = NULL;
	
	// determine the time of day
	time(&now);
	when = *localtime(&now);
	
	C(BestStatusRegGet(cdp->handle, &status));
	if (flag || (status & (STATUS_DATA_COMPARE_ERROR | STATUS_PROTOCOL_ERROR)))
	{
		if (status & (STATUS_DATA_COMPARE_ERROR | STATUS_PROTOCOL_ERROR))
			ret = -1;
		do_out(stream, "\n ***** Card #%ld: (%021x) %s%s%s\t\t %s",
			i, status,
			(status & STATUS_MASTER_ACTIVE ? "running" : "stopped"),
			(status & STATUS_PROTOCOL_ERROR ? ", PROTOCOL_ERROR" : ""),
			(status & STATUS_DATA_COMPARE_ERROR ? ", DATA_COMPARE_ERROR" : ""), asctime (&when));
		if(status & STATUS_PROTOCOL_ERROR) // a protocol error occured
		{
			C(BestObsStatusGet(handle,B_OBS_FIRSTERR,&errval));
			if(cdp->usedboard == B_HW_E2926A || cdp->usedboard == B_HW_E2926A_DEEP)
			{
		        C(BestObsStatusGet(handle,B_OBS_FIRSTERR2,&errval2));
			}
			else 
			{
				errval2 = 0;
			}
			for(j=0;j<32;j++)
			{
				if((1L << j) & errval)
				{
					C(BestObsErrStringGet(handle,j,&perr));
					do_out(stream,"-> Card #%ld: %s\t\t %s",i,perr,asctime(&when));
				}
				if((1L << j) & errval2)
				{
					C(BestObsErrStringGet(handle,j+32,&perr));
					do_out(stream,"-> Card #%ld: The Error was: %s\t\t %s",i,perr,asctime(&when));
				}
			}
		}

	}
	return(ret);
}

//---------------------------------------------------------------------------------------------
// check the status of all cards. If one reports a data compare or protocol error,
// print it out and return -1. Otherwise return 0
// If flag==0, print the result only if a card fails. If flag==1 print results
// unconditionally.
//---------------------------------------------------------------------------------------------
int
check_all_cards(int flag)
{
	unsigned long i;
	card_info_t *cdp;
	int ret = 0;
	
	// check status of all cards
	for (i = 0, cdp = card_info; i < n_cards; i++, cdp++)
	{
		if (check_card(flag, cdp))
			ret = -1;
	}
	return(ret);
}

//---------------------------------------------------------------------------------------------
// ask the user to select a single card from the list of identified cards
//---------------------------------------------------------------------------------------------
card_info_t *
select_single_card(thread_info_t *tip, char *prompt)
{
	unsigned long i;
	card_info_t *cdp;
	b_handletype handle;
	int found = 0;
	char buffer[20];
	struct tm when;
	time_t now;
	
	time(&now);
	when = *localtime(&now);
	
rep:
	
	for (i = 0, cdp = card_info; i < n_cards; i++, cdp++)
	{
		handle = cdp->handle;
		if (!cdp->busy && (cdp->usable == USE))
		{
		//           printf("%s, Card #%d  (bus %d, dev 0x%02lx)  %s\n", asctime (&when),
		//              i, (cdp->devid >> 8) & 0xff, cdp->devid & 0xff, cdp->busy ? "busy":"idle");
			found++;
		}
	}
	
	if (found)
		printf(prompt ? prompt : "Select a card: ");
	else
	{
		do_out(stream, "No cards found.\t\t %s", asctime (&when));
		return(NULL);
	}
	
	gets(buffer);
	i = strtoul(buffer, 0, 0);
	
	if (*buffer == 'q')
		return(NULL);
	
	if (i < 0 || i >= n_cards)
	{
		printf("***** Invalid selection. Valid numbers are in the range: 0...%ld\n", n_cards-1);
		goto rep;
	}
	
	cdp = &card_info[i];
	if (cdp->busy)
	{
		do_out(stream, "***** That card is busy. *****\t\t %s", asctime (&when));
		goto rep;
	}
	
	cdp->busy = 1;
	cdp->tip = tip;
	return(cdp);
}

//---------------------------------------------------------------------------------------------
// Open_all_cards searches the system for E2925A's and E2926A's, opens a 
// connection to them and reads out the assigned memory and I/O ranges.
// After return from this function, n_cards will be set and the card_info array
// will be filled. If an error occurs during init, -1 is returned. Otherwise 0
//---------------------------------------------------------------------------------------------
int
open_all_cards()
{
	unsigned int i;                  // subsys ID (number) for BestDevIdentifierGet()
	card_info_t *cdp;                // points to card info
	b_errtype err;
	b_handletype handle;
	b_hwtype usedboard;
	b_int32 mode;
	int ret = 0;
	struct tm when;
	time_t now;
	char * firmstring;
	
	time(&now);
	
	when = *localtime(&now);
	
	// locate all cards in the system
	// make sure you look for all e2925 and all e2926 cards in your system
	// first look for all E2925A cards in your system
	for (n_cards = 0, i = 0, cdp = card_info; n_cards < MAX_CARDS; i++)
	{
		err = BestDevIdentifierGet(HEWLETT_PACKARD_VEN_ID, E2925A_DEV_ID, i, &(cdp->devid));
		if (err == B_E_NO_BEST_PCI_DEVICE_FOUND)
			break;
		if (err != B_E_OK)
		{
			do_out(stream, "%s BestDevIdentifierGet failed: %s\n", 
				asctime (&when), BestErrorStringGet(err));
			break;
		}
		/* -- to ignore some cards, do something like this:
		if (cdp->devid == 0x70 || cdp->devid == 0x130 || cdp->devid == 0x138)
		continue;
		*/
		do_out(stream, "Card #%d (E2925A) found (bus %d, dev 0x%02lx)\t\t %s",
			n_cards, (cdp->devid >> 8) & 0xff, cdp->devid & 0xff, asctime (&when));
		
		n_cards++;
		cdp++;
	}

	// then look for all E2926A cards in your system
	for (i=0; n_cards < MAX_CARDS; i++)
	{
		err = BestDevIdentifierGet(HEWLETT_PACKARD_VEN_ID, E2926A_DEV_ID, i, &(cdp->devid));
		if (err == B_E_NO_BEST_PCI_DEVICE_FOUND)
			break;
		if (err != B_E_OK)
		{
			do_out(stream, "%s BestDevIdentifierGet failed: %s\t\t", 
				BestErrorStringGet(err),asctime (&when));
			break;
		}
		/* -- to ignore some cards, do something like this:
		if (cdp->devid == 0x70 || cdp->devid == 0x130 || cdp->devid == 0x138)
		continue;
		*/

		do_out(stream, "Card #%d (E2926A) found (bus %d, dev 0x%02lx)\t\t %s",
			n_cards, (cdp->devid >> 8) & 0xff, cdp->devid & 0xff, asctime (&when));

		n_cards++;
		cdp++;
	}

	if (n_cards == 0)
	{
		do_out(stream, "No E2925A and no E2926A found.\t\t %s", asctime (&when));
	}

	// open connections to all the cards and read out their base addresses
	for (i = 0, cdp = card_info; i < n_cards; i++, cdp++)
	{
		// open a connection to the card
		C(BestOpen(&handle, B_PORT_PCI_CONF, cdp->devid));
		if(err != B_E_OK) // we could not open this card, so lets mark it unusable
		{
		  cdp->usable = DONT_USE;
		  continue; 
		}
		cdp->handle = handle;
		cdp->busy = 0;
		cdp->tip = NULL;
		cdp->usable = USE;
		
		C(BestConnect(cdp->handle)); // does not do anything for E2926A, but does not do any harm either
		// show the card number on the display
		C(BestDisplayPropSet(cdp->handle, B_DISP_USER)); // does not do any harm either
		C(BestDisplayWrite(cdp->handle, i));
		
		C(BestGetHWFromHandle(handle,&usedboard)); // gives us the hardware behind this handle
		cdp->usedboard = usedboard; // we want to remember the hardware behind that handle
		
		// Get the firmware version of the card
		BestVersionGet(handle, B_VER_FIRMWARE, &firmstring);
	
		do_out(stream, "Card #%d (%s) firmware version #%s\t\t %s", 
			i, ((usedboard == B_HW_E2925A || usedboard == B_HW_E2925A_DEEP) ? "E2925A" : "E2926A"),
			firmstring, asctime (&when));
 			
		C(BestMasterStop(cdp->handle));                  // just to be on the safe side: stop master
		C(BestStatusRegClear(cdp->handle, 0xffffffff));  // reset data compare errors
		C(BestObsStatusClear(cdp->handle));              // reset protocol errors
		C(BestConfRegGet(cdp->handle, 0x0c, &cacheline_size));

		cacheline_size &= 0xff;
		if (cacheline_size == 0)
		{
			do_out(stream, "***** Card #%d: cacheline size was not set by system BIOS *****\t\t %s",
				i, asctime (&when));
			ret = -1;
		}
		
		// set the master generic properties of the card to default
		C(BestMasterGenPropDefaultSet(cdp->handle));
		C(BestMasterGenPropSet(cdp->handle, B_MGEN_ATTRMODE, B_ATTRMODE_SEQUENTIAL));
		
		// find memory base address of the card, assume it is in decoder #1 (=BAR0)
		C(BestTargetDecoderRead(cdp->handle, 1));
		C(BestTargetDecoderPropGet(cdp->handle, B_DEC_MODE, &mode)); // now called LOCATION, but because
		// of compatibility we can work with MODE
		C(BestTargetDecoderPropGet(cdp->handle, B_DEC_SIZE, &(cdp->memsize)));
		C(BestTargetDecoderPropGet(cdp->handle, B_DEC_BASEADDR, &(cdp->memaddr)));
		
        // check if the decode we are working with is a true 64 bit DAC decoder
		// this only works if we have a E2926A, so check hardware first
		if(usedboard == B_HW_E2926A || usedboard == B_HW_E2926A_DEEP)
		{
			C(BestTargetDecoderPropGet(cdp->handle, B_DEC_DAC, &(cdp->useDAC)));
			if(cdp->useDAC == B_DAC_YES)
			{
				C(BestTargetDecoderPropGet(cdp->handle, B_DEC_BASEADDR_HI, &(cdp->memaddr_hi)));
			}
			else
			{
				cdp->memaddr_hi = 0; // make sure the value is initialized ok
			}
		}
		else
		{
			cdp->useDAC = B_DAC_NO;
			cdp->memaddr_hi = 0; // make sure the values are initialized ok
		}
		
		
		// the dual address capability is not reflected here!! Ask Jim about it.
		
		if (mode != B_MODE_MEM || cdp->memaddr == 0)
		{
			do_out(stream, "***** Card #%d: BAR0 is not a memory base address or not assigned by the BIOS *****\t\t %s",
				i, asctime (&when));
			cdp->memptr = NULL;
			cdp->memaddr = 0;
			cdp->memsize = 0;
			ret = -1;
		}
		else
		{
			cdp->memsize = 1 << cdp->memsize;
			cdp->memptr = (b_int8 *)map_physical_address(cdp->memaddr, cdp->memaddr_hi,
				cdp->memsize, (cdp->devid >> 8) & 0xff, 0);
			if (cdp->memptr == NULL)
			{
				do_out(stream, "***** Card #%d: can't map physical into virtual memory\t\t %s",i, asctime (&when));
			}
			if (dbg)
			{
				if(cdp->useDAC == USE_DAC)
				{
					do_out(stream, "Card #%d: mem base addr (64 bit address): 0x%08lx %08lx (size 0x%1x), virtual addr: 0x%08lx\t\t %s",
						i, cdp->memaddr_hi,cdp->memaddr, cdp->memsize, (unsigned long)cdp->memptr, asctime (&when));
				}
				else
				{
					do_out(stream, "Card #%d: mem base addr: 0x%08lx (size 0x%1x), virtual addr: 0x%08lx\t\t %s", 
						i, cdp->memaddr, cdp->memsize, (unsigned long)cdp->memptr,asctime(&when));
				}

			}
		}
		
		// find I/O base address of the card, assume it is in decoder #2 (=BAR1)
		// or, if the latter one was part of a 64 bit address, look in decoder #3 (=BAR2)
		C(BestTargetDecoderRead(cdp->handle, (cdp->useDAC == USE_DAC ? 3 : 2)));
		C(BestTargetDecoderPropGet(cdp->handle, B_DEC_MODE, &mode));
		C(BestTargetDecoderPropGet(cdp->handle, B_DEC_SIZE, &(cdp->iosize)));
		C(BestTargetDecoderPropGet(cdp->handle, B_DEC_BASEADDR, &(cdp->ioaddr)));
		if (mode != B_MODE_IO || cdp->ioaddr == 0)
		{
			do_out(stream, "***** Card #%d: BAR1 is not an I/O base address or not assigned by the BIOS *****\t\t %s", i, asctime (&when));
			cdp->ioptr = NULL;
			cdp->ioaddr = 0;
			cdp->iosize = 0;
			//ret = -1;
		}
		else
		{
			cdp->iosize = 1 << cdp->iosize;
			// I think we can assume, that the high address for i/o addesses is always 0
			cdp->ioptr = (b_int8 *)map_physical_address(cdp->ioaddr,0, cdp->iosize, (cdp->devid >> 8) & 0xff, 1);
			if (cdp->ioptr == NULL)
			{
				printf("***** Card #%d: can't map I/O space into virtual memory\n", i);
				do_out(stream, "***** Card #%d: can't map I/O space into virtual memory. *****\t\t %s",
					i, asctime (&when));
			}
			if (dbg)
			{
				do_out(stream, "Card #%d: I/O base addr: 0x%08lx (size 0x%1x), virtual addr: 0x%08lx\t\t %s",
					i, cdp->ioaddr, cdp->iosize, (unsigned long)cdp->ioptr, asctime (&when));
			}
		}
		
		// Allocate a block of system memory for each card so that each card
		// has a "playground" for write/read testing
		// Note, that we can reserve a maximum of 4K with a single
		// call, otherwise there is no guarantee, that the memory region is
		// physically contignuous
		
		if (allocate_physical_memory(page_size, &(cdp->sysmem_ptr), &(cdp->sysmem_addr)) != 0)
		{
			do_out(stream, "***** allocating physical memory failed.\t\t %s",asctime(&when));
			cdp->sysmem_ptr = NULL;
			cdp->sysmem_addr = 0;
			ret = -1;
		}
		else
		{
			if (dbg)
			{
				do_out(stream, "Allocated %ldk memory at virtual addr 0x%08lx, physical addr 0x%08lx\n", 
					page_size / 1024, (unsigned long)cdp->sysmem_ptr, (unsigned long)cdp->sysmem_addr);
			}
		}
  }
  
  
  for (i = 0, cdp = card_info; i < n_cards; i++, cdp++)
  {
	  if(cdp->usable == DONT_USE)
		  continue;
	  // clear the observer status register first
	  C(BestObsStatusClear(cdp->handle));
      // run the observer
	  C(BestObsRun(cdp->handle));  // does not do any harm for the E2926A, so leave it here
	  // set up master attribute permutations
	  setup_master_attributes(cdp->handle);
	  // set up target attribute permutations
	  setup_target_attributes(cdp->handle);
	  
	  // set up the analyzer so it runs and triggers with the next protocol error
	  C(BestTracePropSet(handle,B_TRC_HEARTBEATMODE,B_HBMODE_OFF));
	  if(cdp->usedboard != B_HW_E2925A)
		  C(BestTracePropSet(handle,B_TRC_PERFANALYZER_MODE,B_E2925_COMPATIBLE));
	  C(BestTracePattPropSet(handle,B_PT_SQ,"1"));
	  C(BestTracePattPropSet(handle,B_PT_TRIGGER,"berr"));
	  C(BestTraceRun(handle));
	  
  }
  
  return(ret);
}

//-----------------------------------------------------------------------------------
// close_all_cards stops all cards, deallocates any mapped memory and 
// disconnects the cards
//-----------------------------------------------------------------------------------
void close_all_cards()
{
	unsigned int i;
	card_info_t *cdp;
	b_errtype err;
	
	// stop all cards
	for (i = 0, cdp = card_info; i < n_cards; i++, cdp++)
	{
		if(cdp->usable == USE)
			C(BestMasterStop(cdp->handle));
	}
	
	
	// free all allocated memory and disconnect cards
	for (i = 0, cdp = card_info; i < n_cards; i++, cdp++)
	{
		if(cdp->usable == DONT_USE) // skip the cards we are not using
			continue;
		if (cdp->sysmem_ptr)
			free_physical_memory(cdp->sysmem_ptr, page_size);
		if (cdp->memptr)
			unmap_physical_address(cdp->memptr);
		C(BestDisconnect(cdp->handle));
		C(BestClose(cdp->handle));
	}
}



// --------------------------------------------------------------------------------------------
void finish_task(void * tip,char * message,time_t start,struct tm when)
{
	double elapsed;
	time_t end;

	((thread_info_t *)tip)->state = TERMINATED;
	--n_threads;

	do_out(stream, "%s %s terminating...\n", asctime (&when),message);

	time(&end);
	elapsed = difftime(end,start);
	do_out(stream,"Elapsed time of memory read test: %.2f minutes.\n", elapsed / 60.0);
}

//---------------------------------------------------------------------------------------------
// full_test locates all E2925A's in the system, allocates a block of physical memory for each
// card, and instructs the E2925A to write/read/compare.
//---------------------------------------------------------------------------------------------
DWORD __stdcall
stress_test(void *tip)
{
	b_errtype err;
	double elapsed;
	card_info_t *cdp;
	int cont = 1;
	unsigned long i, j;
	const unsigned long size = 256;         // size of blocks that we will transfer
	const unsigned long int_addr1 = 0x200;
	const unsigned long int_addr2 = 0x400;
	const unsigned long int_addr3 = 0x600;
	const unsigned long int_addr4 = 0x800;
	b_int32 tmp;
	volatile unsigned char *xbp;
	//      unsigned char cmp_buf[256];
	struct tm when;
	time_t now, stress_start_run, stress_end_run;
	
	time(&now);
	when = *localtime(&now);
	
	fprintf(stream,"\n%s STRESS TEST\n", asctime(&when));
	time (&stress_start_run);
	for (i = 0, cdp = card_info; i < n_cards; i++, cdp++)
	{
		if (cdp->busy)
			continue;
		if (cdp->memptr == NULL)
			continue;
		cdp->busy = 1;
		cdp->tip = tip;
		// initialize the internal memory of the cards through the mapped address space
		for (j = 0, xbp = ((char *)cdp->memptr) + int_addr1; j < size; j++, xbp++)
			*xbp = 0x5a;
		for (j = 0, xbp = ((char *)cdp->memptr) + int_addr2; j < size; j++, xbp++)
			*xbp = rand();
		//tmp = card_info[(i+1)%n_cards].memaddr;       // write to the next card's built-in memory
		tmp = cdp->sysmem_addr;                                         // write to the allocated system memory
		printf("setup card #%ld to r/w addr 0x%08lx\n", i, tmp);
		fprintf(stream, "%s setup card #%ld to r/w addr 0x%08lx\n", asctime(&when), i, tmp);
		// set up block transfer to read data
		setup_master_block(cdp->handle, BPPR_DIR_WRITE, tmp, 0 , int_addr1, size,
			1, 0, 0, USE_SAC);
		setup_master_block(cdp->handle, BPPR_DIR_READ,  tmp, 0 , int_addr3, size,
			BPPR_BLK_APPEND, 1, int_addr1, USE_SAC);
		setup_master_block(cdp->handle, BPPR_DIR_WRITE, tmp, 0 , int_addr2, size,
			BPPR_BLK_APPEND, 0, 0, USE_SAC);
		setup_master_block(cdp->handle, BPPR_DIR_READ,  tmp, 0 , int_addr3, size,
			BPPR_BLK_APPEND, 1, int_addr2, USE_SAC);
	}
	printf("\n");
	fprintf(stream, "\n");
	// run all cards
	for (i = 0, cdp = card_info; i < n_cards; i++, cdp++)
	{
		if (cdp->tip != tip)
			continue;
		C(BestMasterGenPropSet(cdp->handle, B_MGEN_REPEATMODE, B_REPEATMODE_INFINITE));
		if (cdp->sysmem_addr)
		{
			printf("starting card #%ld\n", i);
			fprintf(stream, "%s Starting card #%ld\n", asctime (&when), i);
			C(BestMasterBlockPageRun(cdp->handle, 1));
		}
	}
	printf("\n");
	fprintf(stream, "\n");
	((thread_info_t *)tip)->state = RUNNING;
	cont = 1;
	while (cont)
	{
		Sleep(500);
		// should now be possible with the E2926A ... 
		//           /* can't write to card in parallel, because of master/target limitations */
		//            for (i = 0, cdp = card_info; i < n_cards; i++, cdp++)
		//            {
		//              for (j = 0, xbp = (char *)cdp->memptr + int_addr4; j < size; j++, xbp++)
		//                cmp_buf[j] = *xbp = rand();
		//              for (j = 0, xbp = (char *)cdp->memptr + int_addr4; j < size; j++, xbp++)
		//              {
		//                if (*xbp != cmp_buf[j])
		//                {
		//                   printf("***** stress_test: card #%d failed with BYTES at offset 0x%08lx\n", i, j);
		//                   fprintf(stream, "***** stress_test: card #%d failed with BYTES at offset 0x%08lx\n", i, j);
		//                   cont = 0;
		//                   break;
		//                 }
		//              }
		//            }
		for (i = 0, cdp = card_info; i < n_cards; i++, cdp++)
		{
			if (cdp->tip != tip)
				continue;
			if (check_card(0, cdp))
				cont = 0;
		}
		
		if (((thread_info_t *)tip)->term_request)
			cont = 0;
		time (&stress_end_run);
		elapsed = difftime(stress_end_run,stress_start_run);
		
		if ((elapsed / 60.0) > ((thread_info_t *)tip)->maxtime)
		{
			break;
		}
	}
	
	((thread_info_t *)tip)->state = TERMINATED;
	--n_threads;
	
	printf("stress_test terminating...\n");
	fprintf(stream, "%s Stress_test terminating...\n", asctime (&when));
	time (&stress_end_run);
	elapsed = difftime(stress_end_run,stress_start_run);
	printf("Elapsed time of stress test: %.2f minutes.\n", elapsed / 60.0);
	fprintf(stream,"Elapsed time of stress test: %.2f minutes.\n", elapsed / 60.0);
	return(0);
}

//---------------------------------------------------------------------------------------------
// mem_test allocates a block of physical memory, fills it with random data (from the CPU
// side) and instructs the E2925A to read the data (through PCI). Afterwards data is uploaded
// from the E2925A and compared with the original data
//---------------------------------------------------------------------------------------------
DWORD __stdcall
mem_write_read_compare(void *tip)
{
	b_handletype handle;
	double elapsed;
	b_errtype err;
	b_int32 tmp;
	unsigned long j;
	unsigned long size;
	card_info_t *cdp;
	volatile unsigned char *xbp;
	const unsigned long int_addr1 = 0x200;
	const unsigned long int_addr2 = 0x400;
	const unsigned long int_addr3 = 0x600;
	struct tm when;
	time_t now, rw_compare_start_run, rw_compare_end_run;
	
	time(&now);
	when = *localtime(&now);
	
	fprintf(stream,"\n%s MEMORY WRITE/READ/COMPARE TO SYSTEM MEMORY\n\n", asctime(&when));
	time (&rw_compare_start_run);
	
	if ((cdp = select_single_card(tip, NULL)) == NULL)
	{
		((thread_info_t *)tip)->state = TERMINATED;
		return 1;
	}
	
	handle = cdp->handle;
	size = 0x200;
	
	// initialize the internal memory of the cards through the mapped address space
	for (j = 0, xbp = ((char *)cdp->memptr) + int_addr1; j < size; j++, xbp++)
		*xbp = 0x5a;
	
	for (j = 0, xbp = ((char *)cdp->memptr) + int_addr2; j < size; j++, xbp++)
		*xbp = rand();
	
	tmp = cdp->sysmem_addr;                                         // write to the allocated system memory
	
	printf("setup card #%d to r/w to physical addr 0x%08lx\n", cdp - card_info, tmp);
	fprintf(stream, "%s Setup card #%d to r/w to physical addr 0x%08lx\n", asctime (&when), cdp - card_info, tmp);
	
	// set up block transfer to write/read/compare data
	setup_master_block(cdp->handle, BPPR_DIR_WRITE, tmp, 0 , int_addr1, size,
		1, 0, 0, USE_SAC);
	setup_master_block(cdp->handle, BPPR_DIR_READ,  tmp, 0 , int_addr3, size,
		BPPR_BLK_APPEND, 1, int_addr1, USE_SAC);
	
	// set up block transfer to write/read/compare data from a different internal address
	setup_master_block(cdp->handle, BPPR_DIR_WRITE, tmp, 0 , int_addr2, size,
		BPPR_BLK_APPEND, 0, 0, USE_SAC);
	setup_master_block(cdp->handle, BPPR_DIR_READ,  tmp, 0 , int_addr3, size,
		BPPR_BLK_APPEND, 1, int_addr2, USE_SAC);
	
	C(BestMasterGenPropSet(cdp->handle, B_MGEN_REPEATMODE, B_REPEATMODE_INFINITE));
	C(BestMasterBlockPageRun(cdp->handle, 1));
	
	((thread_info_t *)tip)->state = RUNNING;
	while (1)
	{
		Sleep(500);
		if (check_card(0, cdp))
			break;
		
		if (((thread_info_t *)tip)->term_request)
			break;
		
		time(&rw_compare_end_run);
		elapsed = difftime(rw_compare_end_run,rw_compare_start_run);
		if ((elapsed / 60.0) > ((thread_info_t *)tip)->maxtime)
		{
			break;
		}
	}
	
	((thread_info_t *)tip)->state = TERMINATED;
	--n_threads;
	printf("memory_write_read_compare terminating...\n");
	fprintf(stream, "%s memory_write_read_compare terminating...\n", asctime (&when));
	time (&rw_compare_end_run);
	elapsed = difftime(rw_compare_end_run,rw_compare_start_run);
	printf("Elapsed time of memory_read_write_compare test: %.2f minutes.\n", elapsed / 60.0);
	fprintf(stream,"Elapsed time of memory_read_write_copare test: %.2f minutes.\n", elapsed / 60.0);
	return(0);
}

//---------------------------------------------------------------------------------------------
// peer2peer locates two E2925A's or E2926A's in the system. One will be used as a master, the other
// one as a target in order to do peer-to-peer traffic on PCI. The two E2925A's don't need
// to be on the same PCI bus. If they are on different buses, the traffic will be sent
// through a PCI-to-PCI bridge.
// peer2peer fills the internal memory of the first card with random data, sets up the
// first card to write to the memory space of the second card, reads the data back
// and compares. Then the same thing is performed with exchanged roles.
//---------------------------------------------------------------------------------------------
DWORD __stdcall
peer2peer(void *tip)
{
	b_errtype err;
	double elapsed;
	unsigned long found = 0;
	int fail = 0;
	b_int32 size;
	card_info_t *cdp1, *cdp2;
	struct tm when;
	time_t now, peer2peer_start_run, peer2peer_end_run;
	
	time(&now);
	when = *localtime(&now);
	fprintf(stream,"\n%s PEER to PEER READ/WRITE TEST\n\n", asctime(&when));
	time (&peer2peer_start_run);
	
	if ((cdp1 = select_single_card(tip, "Select first card:  ")) == NULL)
	{
		((thread_info_t *)tip)->state = TERMINATED;
		--n_threads;
		return 1;
	}
	if ((cdp2 = select_single_card(tip, "Select second card: ")) == NULL)
	{
		((thread_info_t *)tip)->state = TERMINATED;
		--n_threads;
		return 1;
	}
	
	// set size to the minimum of the two memory sizes
	size = cdp1->memsize;
	if (cdp2->memsize < size)
		size = cdp2->memsize;
	if (size > 0x4000)
		size = 0x4000;
	C(BestMasterGenPropDefaultSet(cdp1->handle));
	C(BestMasterGenPropDefaultSet(cdp2->handle));
	
	// if one is a E2925 and the other one requires DAC, we cannot do it
	if(((cdp1->useDAC == USE_DAC) && 
		((cdp2->usedboard == B_HW_E2925A) || (cdp2->usedboard == B_HW_E2925A_DEEP))) ||
	   (((cdp2->useDAC == USE_DAC) && 
		((cdp1->usedboard == B_HW_E2925A) || (cdp1->usedboard == B_HW_E2925A_DEEP)))))
	{
		((thread_info_t *)tip)->state = TERMINATED;
		--n_threads;
	
		printf("Cannot write Dual Address Cycles with an E2925A, sorry, peer2peer terminating...\n");
		fprintf(stream, "Cannot write Dual Address Cycles with an E2925A, sorry, peer2peer teminating...\n");	
		return (1);
	}

	printf("\nUsing cards #%d and #%d for peer-to-peer test\n",
		cdp1 - card_info, cdp2 - card_info);
	printf("block size: 0x%lx\n", size);
	
	fprintf(stream, "\n%s Using cards #%d and #%d for peer-to-peer test.\n",
		asctime (&when), cdp1 - card_info, cdp2 - card_info);
	fprintf(stream, "block size: 0x%lx\n", size);

	if(cdp1->useDAC == USE_DAC || cdp2->useDAC == USE_DAC)
	{
		printf("physical address: card %d:0x",cdp1 - card_info);
		fprintf(stream,"physical address: card %d:0x",cdp1 -card_info);
		if(cdp1->useDAC == USE_DAC)
		  {
		    printf("%08lx ",cdp1->memaddr_hi);
		    fprintf(stream,"%08lx ",cdp1->memaddr_hi);
		  }
		printf("%08lx, card %d:0x", cdp1->memaddr, cdp2 - card_info);
		fprintf(stream,"%08lx, card %d:0x", cdp1->memaddr, cdp2 - card_info);
		if(cdp2->useDAC == USE_DAC)
		  {
		    printf("%08lx ",cdp2->memaddr_hi);
		    fprintf(stream,"%08lx ",cdp2->memaddr_hi);
		  }
		printf("%08lx\n", cdp2->memaddr);
		fprintf(stream,"%08lx\n", cdp2->memaddr);
	}
	else
	{
		printf("physical address: card %d:0x%08lx, card %d:0x%08lx\n",
			cdp1 - card_info, cdp1->memaddr, cdp2 - card_info, cdp2->memaddr);
		fprintf(stream, "physical address: card %d:0x%08lx, card %d:0x%08lx\n",
			cdp1 - card_info, cdp1->memaddr, cdp2 - card_info, cdp2->memaddr);
	}

	
	// now do 64 bit addressing (just in case we talk to E2926A's only)

	// setup card1 to write/read/compare card2's memory
	setup_master_block(cdp1->handle, BPPR_DIR_WRITE, cdp2->memaddr, cdp2->memaddr_hi , 
		0x10000, size, 1, 0, 0, cdp2->useDAC);
	setup_master_block(cdp1->handle, BPPR_DIR_READ,  cdp2->memaddr, cdp2->memaddr_hi , 
		0x14000, size, BPPR_BLK_APPEND, 1, 0x10000, cdp2->useDAC);
	// make it twice (shifted by one dword)
	setup_master_block(cdp1->handle, BPPR_DIR_WRITE, cdp2->memaddr, cdp2->memaddr_hi , 
		0x10004, size, BPPR_BLK_APPEND, 0, 0, cdp2->useDAC);
	setup_master_block(cdp1->handle, BPPR_DIR_READ,  cdp2->memaddr, cdp2->memaddr_hi , 
		0x14000, size, BPPR_BLK_APPEND, 1, 0x10004, cdp2->useDAC);

	C(BestMasterGenPropSet(cdp1->handle, B_MGEN_REPEATMODE, B_REPEATMODE_INFINITE));
	C(BestMasterBlockPageRun(cdp1->handle, 1));
	
	if(((cdp1->usedboard == B_HW_E2926A) || (cdp1->usedboard == B_HW_E2926A_DEEP)) &&
		((cdp2->usedboard == B_HW_E2926A) || (cdp2->usedboard == B_HW_E2926A_DEEP)))
	{
		// can run the other card in parallel with the E2926A (the E2925A had a bug there)
		setup_master_block(cdp2->handle, BPPR_DIR_WRITE, cdp1->memaddr, cdp1->memaddr_hi ,
			0x10000, size, 1, 0, 0, cdp1->useDAC);
		setup_master_block(cdp2->handle, BPPR_DIR_READ,  cdp1->memaddr, cdp1->memaddr_hi , 
			0x14000, size, BPPR_BLK_APPEND, 1, 0x10000, cdp1->useDAC);
		setup_master_block(cdp2->handle, BPPR_DIR_WRITE, cdp1->memaddr, cdp1->memaddr_hi , 
			0x10004, size, BPPR_BLK_APPEND, 0, 0, cdp1->useDAC);
		setup_master_block(cdp2->handle, BPPR_DIR_READ,  cdp1->memaddr, cdp1->memaddr_hi , 
			0x14000, size, BPPR_BLK_APPEND, 1, 0x10004, cdp1->useDAC);
		
		C(BestMasterGenPropSet(cdp2->handle, B_MGEN_REPEATMODE, B_REPEATMODE_INFINITE));
		C(BestMasterBlockPageRun(cdp2->handle, 1));
	}

	
	((thread_info_t *)tip)->state = RUNNING;
	while (1)
	{
		Sleep(500);
		if (check_card(0, cdp1))
			break;
		if (check_card(0, cdp2))
			break;
		if (((thread_info_t *)tip)->term_request)
			break;
		time(&peer2peer_end_run);
		elapsed = difftime(peer2peer_end_run, peer2peer_start_run);
		if ((elapsed / 60.0) > ((thread_info_t *)tip)->maxtime)
		{
			break;
		}
	}
	
	((thread_info_t *)tip)->state = TERMINATED;
	--n_threads;
	
	printf("peer2peer terminating...\n");
	fprintf(stream, "peer2peer teminating...\n");
	
	time (&peer2peer_end_run);
	elapsed = difftime(peer2peer_end_run,peer2peer_start_run);
	
	printf("Elapsed time of peer2peer test: %.2f minutes.\n", elapsed / 60.0);
	fprintf(stream,"Elapsed time of peer2peer test: %.2f minutes.\n", elapsed / 60.0);
	
	return(0);
}

//---------------------------------------------------------------------------------------------
// config_scan scans the PCI bus in which the E2925A is plugged in. It reads the vendor ID/
// device ID register and the class code and prints it out
//---------------------------------------------------------------------------------------------
DWORD __stdcall
config_scan(void *tip)
{
	b_handletype handle;
	double elapsed;
	b_errtype err;
	b_int32 val, class_code;
	unsigned long i, j, found = 0;
	char *s;
	card_info_t *cdp;
	struct tm when;
	time_t now, config_scan_start_run, config_scan_end_run;
	
	time(&now);
	when = *localtime(&now);
	
	fprintf(stream,"\n%s PCI CONFIG SCAN TEST\n\n", asctime(&when));
	time (&config_scan_start_run);
	
	if ((cdp = select_single_card(tip, NULL)) == NULL)
	{
		((thread_info_t *)tip)->state = TERMINATED;
		return 1;
	}
	handle = cdp->handle;
	
	for (i = 0x800; i; i <<= 1)                   // loop through all devices on this PCI bus
	{
		for (j = 0; j < 0x800; j += 0x100) // loop through all functions within a device
		{
			// read vendor ID/device ID register (config space offset 0)
			err = BestHostPCIRegGet(handle, B_ASP_CONFIG, i+j+0, &val, 4);
			if (err == B_E_MASTER_ABORT || val == 0xffffffff)
			{
				// master abort or 0xffffffff is normal for non-existing devices
			}
			else if (err != B_E_OK)
			{
				printf("BestHostPCIRegGet failed (%s)\n", BestErrorStringGet(err));
				fprintf(stream, "%s BestHostPCIRegGet failed (%s)\n", asctime (&when), BestErrorStringGet(err));
			}
			else
			{
				// read class code register (config space offset 8)
				C(BestHostPCIRegGet(handle, B_ASP_CONFIG, i+j+8, &class_code, 4));
				// class code is in bits 8...31
				class_code >>= 8;
				switch(class_code & 0xff0000)
				{
				case 0x000000: s = "Device was built before class codes were defined"; break;
				case 0x010000: s = "Mass storage controller"; break;
				case 0x020000: s = "Network controller"; break;
				case 0x030000: s = "Display controller"; break;
				case 0x040000: s = "Multimedia controller"; break;
				case 0x050000: s = "Memory controller"; break;
				case 0x060000: s = "Bridge device"; break;
				case 0x070000: s = "Simple communication controller"; break;
				case 0x080000: s = "Base system peripheral"; break;
				case 0x090000: s = "Input device"; break;
				case 0x0a0000: s = "Docking station"; break;
				case 0x0b0000: s = "Processor"; break;
				case 0x0c0000: s = "Serial bus controller"; break;
				case 0xff0000: s = "Device does not fit into any defined class"; break;
				default:       s = "Reserved class code"; break;
				}
				if (found == 0)
				{
					printf("Cfg.Addr VendorID DeviceID Class\n");
					printf("--------------------------------\n");
					printf("Current time is: %s\n", asctime (&when));
					printf("-----------------------------------------\n");
					fprintf(stream, "Cfg.Addr VendorID DeviceID Class\n");
					fprintf(stream, "--------------------------------\n");
					fprintf(stream, "Current time is: %s\n", asctime (&when));
					fprintf(stream, "-----------------------------------------\n");
				}
				printf("%08lx   %04lx     %04lx   %s\n", i+j, val >> 16, val & 0xffff, s);
				fprintf(stream,"%08lx   %04lx     %04lx   %s\n", i+j, val >> 16, val & 0xffff, s);
				// Read cache line size (config space offset 0x0C)
				
				err = BestHostPCIRegGet (handle, B_ASP_CONFIG, i+j+0x0C, &val, 1);
				if (err == B_E_MASTER_ABORT || val == 0xFFFFFFFF)
				{
					// Master abort or 0xFFFFFFFF is normal for non-existing devices.
				}
				else
				{ 
					printf("\nThe cache line size is: %02lx.\n\n", val);
					fprintf(stream, "\nThe cache line size is: %02lx.\n\n", val);
				}
				// Read Base Addr register 0 (config space offset 0x10)
				err = BestHostPCIRegGet (handle, B_ASP_CONFIG, i+j+0x10, &val, 4);
				if (err == B_E_MASTER_ABORT || val == 0xFFFFFFFF)
				{
					// Master abort or 0xFFFFFFFF is normal for non-existing devices.
				}
				else
				{
					// Bit 0:  Address space
					
					printf ("According to the Base Address register 0:\n"); 
					fprintf (stream, "According to the Base Address register 0:\n");
					if (val & 0x0001 == 0x1)
					{
						printf ("  Address space enabled.\n");
						fprintf (stream, "  Address space enabled.\n");
					}
					else 
						printf ("  Address Space disabled.\n");
					fprintf (stream, "  Address Space disabled.\n");
					// Bit 1:  Memory space location
					if (val & 0x0002 == 0x1)
					{
						printf ("  Memory space location enabled.\n");
						fprintf (stream, "  Memory space location enabled.\n");
					}
					else
						printf ("  Memory space location disabled.\n");
					fprintf (stream, "  Memory space location disabled.\n");
					// Bit 2:  Memory space location
					if (val & 0x0004 == 0x1) 
					{
						printf ("  Memory space location enabled.\n");
						fprintf (stream, "  Memory space location enabled.\n");
					}
					else 
						printf ("  Memory space location disabled.\n");
					fprintf (stream, "  Memory space location disabled.\n");
				}
				// Read Base Addr register 1 (config space ofset 0x14)
				err = BestHostPCIRegGet (handle, B_ASP_CONFIG, i+j+0x14, &val, 4);
				if (err == B_E_MASTER_ABORT || val == 0xFFFFFFFF)
				{
					// Master abort or 0xFFFFFFFF is normal for non-existing devices.
				}
				else
				{
					// Bit 0:  Address space
					printf ("According to the Base Address register 1:\n");
					fprintf (stream, "According to the Base Address register 1:\n");
					if (val & 0x0001 == 0x1) 
					{
						printf ("  Address Space enabled.\n");
						fprintf (stream, "  Address Space enabled.\n");
					}
					else 
						printf ("  Address Space disabled.\n");
					fprintf (stream, "  Address Space disabled.\n");
				}
				
				// Read Subsystem ID (config space offset 0x2E)
				err = BestHostPCIRegGet (handle, B_ASP_CONFIG, i+j+0x2E, &val, 2);
				if (err == B_E_MASTER_ABORT || val == 0xFFFFFFF)
				{
					// Master abort of 0xFFFFFFFF is normal for non-existing devices.
				}
				else
					printf ("\nThe subsystem ID is: %04lx.\n\n", val);
				fprintf (stream, "\nThe subsystem ID is %04lx.\n\n", val);
				
				// Read Command Register (config space ofset 0x04)
				err = BestHostPCIRegGet (handle, B_ASP_CONFIG, i+j+0x04, &val, 2);
				if (err == B_E_MASTER_ABORT || val == 0xFFFFFFFF)
				{
					// Master abort or 0xFFFFFFFF is normal for non-existing devices.
				}
				else
				{
					// Bit 0: I/O Space Control, RW
					if (val & 0x0001 == 0x1) {
						printf ("  IO Space Control is enabled.\n");
						fprintf (stream, "  IO Space Control is enabled.\n");
					}
					else
						printf ("  IO Space Control is disabled.\n");	   
					fprintf (stream, "  IO Space Control is disabled.\n");
					
					// Bit 1: Memory space, RW
					if (val & 0x0002 == 0x1) {
						printf ("  Memory space is enabled.\n");
						fprintf (stream, "  Memory space is enabled.\n");
					}
					else
						printf ("  Memory space is disabled.\n");
					fprintf (stream, "  Memory space is disabed.\n");
					
					// Bit 2: Bus Master Control, RW
					if (val & 0x0004 == 0x1) {
						printf ("  Bus Master Control is enabled.\n");
						fprintf (stream, "  Bus Master Control is enabled.\n");
					}
					else
						printf ("  Bus Master Control is disabled.\n");
					fprintf (stream, "  Bus Master Control is disabled.\n");
					
					// Bit 4: Memory Write and Invalidate Enable, RW
					if (val & 0x0008 == 0x1) {
						printf ("  Memory Write and Invalidate is enabled.\n"); 
						fprintf (stream, "  Memory Write and Invalidate is enabled.\n");
					}
					else
						printf ("  Memory Write and Invalidate is disabled.\n");
					fprintf (stream, "  Memory Write and Invalidate is disabled.\n");
					
					// Bit9: Master fast back to back is enabled, RO
					if (val & 0x0100 == 0x1) {
						printf ("  Master fast back to back is enabled.\n");
						fprintf (stream, "  Master fast back to back is enabled.\n");
					}
					else
						printf ("Master fast back to back is disabled.\n");
					fprintf (stream, "  Master fast back to back is disabled.\n");
				}
				found++;
				// if we are looking at function #0 of a given device, check whether this
				// is a multifunction device by reading the header type at offset 0x0e
				// (Note the 5th parameter in BestHostPCIRegGet: reading a single byte)
				// If it is a single function device, skip the other functions
				if (j == 0)
				{
					C(BestHostPCIRegGet(handle, B_ASP_CONFIG, i+0x0e, &val, 1));
					if ((val & 0x80) == 0)
						break;
				}
      }
    }
  }
  if (found == 0)
  {
	  printf("No PCI devices found\n");
	  fprintf(stream, "No PCI devices found\n");
  }
  
  cdp->busy = 0;
  ((thread_info_t *)tip)->state = TERMINATED;
  --n_threads;
  time (&config_scan_end_run);
  elapsed = difftime(config_scan_end_run,config_scan_start_run);
  
  printf("Elapsed time of config_scan test: %.2f minutes.\n", elapsed / 60.0);
  fprintf(stream,"Elapsed time of config scan test: %.2f minutes.\n", elapsed / 60.0);
  
  return(0);
}

//---------------------------------------------------------------------------------------------
// a serial test (nothing done yet)
//---------------------------------------------------------------------------------------------
DWORD __stdcall
serial_test(void *tip)
{
	struct tm when;
	time_t now;
	time(&now);
	when = *localtime(&now);
	fprintf(stream,"\n%s SERIAL LOOPBACK TEST\n\n", asctime(&when));
	((thread_info_t *)tip)->state = TERMINATED;
	return(0);
}

//---------------------------------------------------------------------------------------------
// CPU accesses I/O space on HPBEST card
//---------------------------------------------------------------------------------------------
DWORD __stdcall
cpu_io_test(void *tip)
{
	//b_errtype err;
	card_info_t *cdp;
	double elapsed;
	unsigned long *cmp_buf;
	b_int32 size;
	b_int32 i;
	int cont = 1;
	b_handletype handle;
	unsigned char *xbp;
	unsigned long xl;
	struct tm when;
	time_t now, cpu_IO_end_run, cpu_IO_start_run;
	
	time(&now);
	when = *localtime(&now);
	
	fprintf(stream,"\n%s CPU ACCESS to HPBEST I/O SPACE\n\n", asctime(&when));
	time (&cpu_IO_start_run);
	
	if ((cdp = select_single_card(tip, NULL)) == NULL)
	{
		((thread_info_t *)tip)->state = TERMINATED;
		return 1;
	}
	
	handle = cdp->handle;
	size = cdp->iosize;
	if (size > 16)
		size = 16;
	printf("block size = 0x%lx\n", size);
	fprintf(stream, "block size = 0x%lx\n", size);
	printf("virtual addr = 0x%08lx, physical = 0x%08lx\n", (DWORD)cdp->ioptr, cdp->ioaddr);
	fprintf(stream, "virtual addr = 0x%08lx, physical = 0x%08lx\n", (DWORD)cdp->ioptr, cdp->ioaddr);
	cmp_buf = malloc(size * sizeof(unsigned long));
	if (cdp->ioptr)
	{
		((thread_info_t *)tip)->state = RUNNING;
		while (cont)
		{
			//                      Sleep(500);
			for (i = 0,  xbp = cdp->ioptr; i < size; i++, xbp += 1)
			{
				xl = ((unsigned char *)cmp_buf)[i] = rand();
				io_access(BEST_NTMEM_IO_WRITE_BYTE, (unsigned long)xbp, &xl);
			}
			for (i = 0,  xbp = cdp->ioptr; i < size; i++, xbp += 1)
			{
				io_access(BEST_NTMEM_IO_READ_BYTE, (unsigned long)xbp, &xl);
				if (xl != ((unsigned char *)cmp_buf)[i])
				{
					printf("***** cpu_io_test: failed with BYTES at offset 0x%08lx\n", i);
					fprintf(stream,"***** cpu_io_test: failed wait BYTES at offset 0x%08lx\n", i);
					cont = 0;
					break;
				}
			}
			for (i = 0,  xbp = cdp->ioptr; i < size / 2; i++, xbp += 2)
			{
				xl = ((unsigned short *)cmp_buf)[i] = rand();
				io_access(BEST_NTMEM_IO_WRITE_WORD, (unsigned long)xbp, &xl);
			}
			for (i = 0,  xbp = cdp->ioptr; i < size / 2; i++, xbp += 2)
			{
				io_access(BEST_NTMEM_IO_READ_WORD, (unsigned long)xbp, &xl);
				if (xl != ((unsigned short *)cmp_buf)[i])
				{
					printf("***** cpu_io_test: failed with WORDS at offset 0x%08lx\n", i);
					fprintf(stream,"***** cpu_io_test: failed wait WORDS at offset 0x%08lx\n", i);
					cont = 0;
					break;
				}
			}
			for (i = 0,  xbp = cdp->ioptr; i < size / 4; i++, xbp += 4)
			{
				xl = ((unsigned long *)cmp_buf)[i] = rand();
				io_access(BEST_NTMEM_IO_WRITE_DWORD, (unsigned long)xbp, &xl);
			}
			for (i = 0,  xbp = cdp->ioptr; i < size / 4; i++, xbp += 4)
			{
				io_access(BEST_NTMEM_IO_READ_DWORD, (unsigned long)xbp, &xl);
				if (xl != ((unsigned long *)cmp_buf)[i])
				{
					printf("***** cpu_io_test: failed with DWORDS at offset 0x%08lx\n", i);
					fprintf(stream,"***** cpu_io_test: failed wait DWORDS at offset 0x%08lx\n", i);
					cont = 0;
					break;
				}
			}
			if (check_card(0, cdp))
				break;
			
			if (((thread_info_t *)tip)->term_request)
				break;
			
			time(&cpu_IO_end_run);
			elapsed = difftime(cpu_IO_end_run, cpu_IO_start_run);
			
			if ((elapsed / 60.0) > ((thread_info_t *)tip)->maxtime)
			{
				break;
			}
		}
	}
	else
	{
		printf("***** can't access on-board I/O space\n");
		fprintf(stream, "%s ***** Can't access on-board I/O space.\n", asctime (&when));
	}
	
	free(cmp_buf);
	printf("cpu_io_test terminating...\n");
	fprintf(stream, "cpu_io_test terminating...\n");
	
	((thread_info_t *)tip)->state = TERMINATED;
	--n_threads;
	
	time (&cpu_IO_end_run);
	elapsed = difftime(cpu_IO_end_run,cpu_IO_start_run);
	printf("Elapsed time of cpu_io_test: %.2f minutes.\n", elapsed / 60.0);
	fprintf(stream,"Elapsed time of cpu_io_test: %.2f minutes.\n", elapsed / 60.0);
	return(0);
}

//---------------------------------------------------------------------------------------------
// CPU accesses memory space on HPBEST card
//---------------------------------------------------------------------------------------------
DWORD __stdcall
cpu_mem_test(void *tip)
{
	card_info_t *cdp;
	double elapsed;
	unsigned long *cmp_buf;
	b_int32 size, i;
	int cont = 1;
	b_handletype handle;
	volatile unsigned char *xbp;
	volatile unsigned short *xsp;
	volatile unsigned long *xlp;
	struct tm when;
	time_t now,cpu_start_run,cpu_end_run;
	
	time(&now);
	when = *localtime(&now);
	
	fprintf(stream,"\n%s CPU ACCESS to HPBEST MEMORY SPACE\n\n", asctime(&when));
	time (&cpu_start_run);
	
	if ((cdp = select_single_card(tip, NULL)) == NULL)
	{
		((thread_info_t *)tip)->state = TERMINATED;
		return 1;
	}
	handle = cdp->handle;
	size = cdp->memsize;
	if (size > page_size)
		size = page_size;
	size = 16;
	
	printf("block size = 0x%lx\n", size);
	fprintf(stream, "block size = 0x%lx\n", size);
	printf("virtual addr = 0x%08lx, physical = 0x%08lx\n", (DWORD)cdp->memptr, cdp->memaddr);
	fprintf(stream, "virtual addr = 0x%08lx, physical = 0x%08lx\n", (DWORD)cdp->memptr, cdp->memaddr);
	
	cmp_buf = malloc(4*size);
	if (cdp->memptr)
	{
		((thread_info_t *)tip)->state = RUNNING;
		while (cont)
		{
			for (i = 0, xbp = cdp->memptr; i < size; i++, xbp++)
				cmp_buf[i] = *xbp = rand();
			for (i = 0, xbp = cdp->memptr; i < size; i++, xbp++)
			{
				if (*xbp != cmp_buf[i])
				{
					printf("***** cpu_mem_test: failed with BYTES at offset 0x%08lx\n", i);
					fprintf(stream,"%s ***** cpu_mem_test: failed with BYTES at offset 0x%08lx\n", asctime (&when), i);
					cont = 0;
					break;
				}
			}
			for (i = 0, xsp = cdp->memptr; i < size / 2; i++, xsp++)
				cmp_buf[i] = *xsp = rand();
			for (i = 0, xsp = cdp->memptr; i < size / 2; i++, xsp++)
			{
				if (*xsp != cmp_buf[i])
				{
					printf("***** cpu_mem_test: failed with WORDS at offset 0x%08lx\n", i);
					fprintf(stream,"%s ***** cpu_mem_test: failed with WORDS at offset 0x%08lx\n", asctime (&when), i);
					cont = 0;
					break;
				}
			}
			for (i = 0, xlp = cdp->memptr; i < size / 4; i++, xlp++)
				cmp_buf[i] = *xlp = (rand() | rand() << 16);
			for (i = 0, xlp = cdp->memptr; i < size / 4; i++, xlp++)
			{
				if (*xlp != cmp_buf[i])
				{
					printf("***** cpu_mem_test: failed with DWORDS at offset 0x%08lx\n", i);
					fprintf(stream,"%s ***** cpu_mem_test: failed with DWORDS at offset 0x%08lx\n", asctime (&when), i);
					cont = 0;
					break;
				}
			}
			if (check_card(0, cdp))
				break;
			
			if (((thread_info_t *)tip)->term_request)
				break;
			
			Sleep(500);
			
			time(&cpu_end_run);
			elapsed = difftime(cpu_end_run, cpu_start_run);
			if ((elapsed / 60.0) > ((thread_info_t *)tip)->maxtime)
			{
				break;
			}
		}
	}
	else
	{
		printf("***** can't access on-board memory\n");
		fprintf(stream, "%s ***** Can't access on-board memory.\n", asctime (&when));
	}
	
	free(cmp_buf);
	
	cdp->busy = 0;
	
	do_out(stream, "%s cpu_mem_test terminating...\n", asctime (&when));

	((thread_info_t *)tip)->state = TERMINATED;
	--n_threads;
	time (&cpu_end_run);
	elapsed = difftime(cpu_end_run,cpu_start_run);
	do_out(stream,"Elapsed time of cpu_mem_test: %.2f minutes.\n", elapsed / 60.0);

	return(0);
}

//---------------------------------------------------------------------------------------------
// HPBEST reads from system memory
//---------------------------------------------------------------------------------------------
DWORD __stdcall
mem_read(void *tip)
{
	b_errtype err;
	double elapsed;
	card_info_t *cdp;
	char buffer[20];
	b_int32 addr, len;
	b_handletype handle;
	struct tm when;
	time_t now, mem_start_run, mem_end_run;
	
	time(&now);
	when = *localtime(&now);
	
	fprintf(stream,"\n%s BUS MASTER MEMORY READ TEST\n\n", asctime(&when));
	time (&mem_start_run);
	
	if ((cdp = select_single_card(tip, NULL)) == NULL)
	{
		((thread_info_t *)tip)->state = TERMINATED;
		return 1;
	}
	
	handle = cdp->handle;
	printf("Enter base address of memory to read: ");
	gets(buffer);
	addr = strtoul(buffer, 0, 0);
	printf("Enter block length of memory read (in bytes):    ");
	gets(buffer);
	len = strtoul(buffer, 0, 0);
	
	if(len == 0)
	{
		printf("A block length of 0 bytes is not a valid length. Change it to 4 bytes.\n");
		len = 4;
	}

	C(BestMasterGenPropDefaultSet(cdp->handle));
	C(BestMasterGenPropSet(cdp->handle, B_MGEN_REPEATMODE, B_REPEATMODE_INFINITE));
	
	setup_master_block(cdp->handle, BPPR_DIR_READ, addr, 0 , 0x0000, len, 1, 0, 0, USE_SAC);
	C(BestMasterBlockPageRun(cdp->handle, 1));
	
	printf("mem_read running...\n");
	fprintf(stream, "%s Mem_read_running...\n", asctime (&when));
	((thread_info_t *)tip)->state = RUNNING;
	
	while (1)
	{
		Sleep(500);
		if (check_card(0, cdp))
			break;
		if (((thread_info_t *)tip)->term_request)
			break;
		time(&mem_end_run);
		elapsed = difftime(mem_end_run,mem_start_run);
		if ((elapsed / 60.0) > ((thread_info_t *)tip)->maxtime)
		{
			break;
		}
	}
	
	C(BestMasterStop(cdp->handle));
	cdp->busy = 0;

	((thread_info_t *)tip)->state = TERMINATED;
	--n_threads;

	do_out(stream, "%s Mem_read terminating...\n", asctime (&when));
	time(&mem_end_run);
	elapsed = difftime(mem_end_run,mem_start_run);
	do_out(stream,"Elapsed time of memory read test: %.2f minutes.\n", elapsed / 60.0);
	return(0);
}


// -----------------------------------------------------------------------------------------
/* starts a new thread for every card and task found */
// -----------------------------------------------------------------------------------------
int
spawn(fct_tab_t *ftp)
{
	thread_info_t *tip;
	struct tm when;
	time_t now;
	char buf[20];
	
	time(&now);
	when = *localtime(&now);
	
	if (n_threads >= MAX_THREADS)
	{
		do_out(stream,"%s Can't open more threads.\n", asctime (&when));
		return(-1);
	}
	else
	{
		tip = &thread_info[n_threads];
		printf("Number of minutes to run this test? ");
		gets(buf);
		if (*buf)
			tip->maxtime = atof(buf);
		else
			tip->maxtime = default_maxtime;
		
		//printf("Test will run for %g minutes\n", tip->maxtime);
		tip->state = STARTING;
		tip->term_request = 0;
		tip->ftp = ftp;
		tip->hThread = CreateThread(NULL, 4096, ftp->fct, tip, 0, &(tip->idThread));
		while (tip->state == STARTING)
			Sleep(100);
		// just in case the thread terminated immediately, allow some time for messages
		Sleep(1000);
		n_threads++;
	}
}

// --------------------------------------------------------------------------------------------
// the list of the functions supported by this program
//---------------------------------------------------------------------------------------------
fct_tab_t fct_tab[] = {
	{ "Bus Master Memory Read", mem_read },
	{ "Memory Write/Read/Compare to system memory", mem_write_read_compare },
	{ "CPU access to HPBEST memory space", cpu_mem_test },
	{ "CPU access to HPBEST I/O space", cpu_io_test },
	{ "Serial loopback test", serial_test },
	{ "Peer to peer read/write test", peer2peer },
	{ "PCI Configuration scan", config_scan },
	{ "Stress test", stress_test },
	{ NULL, NULL }
};


/* --------------------------------------------------------------------------------------- */
/* ------------------------------- main function for bustest ----------------------------- */
/* --------------------------------------------------------------------------------------- */
int main(int argc, char **argv)
{
	b_errtype err = B_E_OK;
	unsigned int selection = 0;
	char buf[20];
	int cont = 1;
	unsigned long i = 0;
	int j = 0;
	double elapsed = 0.0;
	thread_info_t *tip;
	DWORD status = 0;
	card_info_t *cdp;
	struct fct_tab_s *ftp;
	struct tm when;
	time_t now, start_run, end_run;
	
	// use the system to copy the log file to another location
	system("copy BUS_TEST.LOG BUS_TEST.BAK");
	
	// open a new log file
	if ((stream = fopen("BUS_TEST.LOG", "w")) == NULL)
	{
		printf ("Couldn't open log file.\n");
	}
	
	// welcome heading
	do_out(stream,"\nE2925A/E2926A sample tests (Version: %s %s)\n", __DATE__, __TIME__);
	do_out (stream, "Version 2.01 (supports E2925A and E2926A), March 17th, 1998\n");
	
	time(&now);
	when = *localtime(&now);
	
	do_out(stream, "Current time is %s\n",asctime (&when));
	time (&start_run);
	
	for (j = 1; j < argc; j++)
	{
		if (strcmp(argv[j], "-d") == 0) dbg = 1;
	}
	
	if (open_all_cards() == 0) // open all cards and prepare the info structure 
	{
		while (cont)
		{
			printf("\n-------------------------------------------------------------------------\n");
			for (i = 0, tip = thread_info; i < n_threads; i++, tip++)
			{
				GetExitCodeThread(tip->hThread, &status);
				
				do_out(stream,"Thread #%d %s  (%s)\n",
					i, status == STILL_ACTIVE ? "active":"terminated", tip->ftp->name);
			}
			fprintf(stream, "\n");
			for (i = 0, cdp = card_info; i < n_cards; i++, cdp++)
			{
			    if(cdp->usable == DONT_USE)
					continue; // skip unusable cards

				if (cdp->tip)
				{
					GetExitCodeThread(cdp->tip->hThread, &status);
					if (status != STILL_ACTIVE)
					{
						C(BestMasterStop(cdp->handle));
						cdp->busy = 0;
						cdp->tip = NULL;
					}
				}
			
				do_out(stream,"Card #%d  (bus %d, dev 0x%02lx)  %s",
					i, (cdp->devid >> 8) & 0xff, cdp->devid & 0xff, cdp->busy ? "busy":"idle");
			
				if (cdp->tip)
				{
					do_out(stream,", used by thread #%d", cdp->tip - thread_info);
				}
				do_out(stream,"\n");
			}

			do_out(stream,"\n");
			
			for (i = 1, ftp = fct_tab; ftp->name; i++, ftp++)
			{
				do_out(stream,"  %d - %s\n", i, ftp->name);
			}
			printf("  q - Exit\n");
			printf("\nSelect test: ");

			gets(buf);
			selection = atoi(buf);
			printf("\n");

			if (*buf == 'q')
				cont = 0;
			else
			{
				if (selection >= 1 && selection < i)
					spawn(&fct_tab[selection - 1]);
			}
		}

		if(n_threads) // do the following only if threads are pending
		{
			do_out(stream,"Waiting for %d threads to terminate...\n", n_threads);
			
			for (i = 0; i < n_threads; i++)
				thread_info[i].term_request = 1;
			
			if(n_threads)   // wait only if you have to 
				Sleep(2000);    // allow some time for the threads to terminate
			for (i = 0, tip = thread_info; i < n_threads; i++, tip++)
			{
				GetExitCodeThread(tip->hThread, &status);
				if (status == STILL_ACTIVE)
				{
					do_out(stream,"Killing thread #%d\n", i);
					TerminateThread(tip->hThread, 0);
				}
			}
		}
		else
		{
			do_out(stream,"No threads left, closing all cards...\n\n");
		}

		close_all_cards();
	}
	
	//printf("press RETURN to continue");
	//gets(buf);
	
	time(&now);
	
	when = *localtime(&now);
	time(&end_run);
	
	do_out(stream, "Ending time is %s",asctime (&when));
	
	elapsed = difftime(end_run,start_run);
	
	do_out(stream,"Elapsed time of tests(s): %.2f minutes.\n", elapsed / 60.0);

	fclose(stream);
	
	return(0);
}
